/*global define */
/*jslint white: true */

/*
	Graphics:

	This object stores vector graphics used for display.
*/

define ([   "src/build/Node",   "src/build/Path",   "src/build/CommandPlayer",  "src/utils"],
function(   Node,               Path,               CommandPlayer,              utils) {
	'use strict';

	var Node_clone = Node.prototype.clone;
	
	function Graphic(inName) {
		this.Node("Graphics", inName);
		this.pGraphics = { commands : [] };
	}

	utils.mixin(Graphic, Node);

	function PrintPlayer (inGraphics) {
		var that = this, m_graphics = inGraphics, m_prints = "", printPath, printCmd;

		printPath = function (inPts) {
			var p;

			m_prints += "[";
			for (p = 0; p < inPts.length; p += 1) {
				if (p !== 0) {
					m_prints += ",";
				}
				m_prints += inPts[p].toString();
			}
			m_prints += "]";
		};

		printCmd = function (inFuncName) {
			if (m_prints !== "") {
				m_prints += ", ";
			}
			m_prints += inFuncName;
		};

		that.fillPath = function (inParameters) {
			printCmd("fillPath", inParameters);
			if (inParameters !== undefined) {
				m_prints += " : color[";
				m_prints += inParameters.color.print();
				m_prints += "], path[";
				m_prints += inParameters.path.print();
				m_prints += "]";
			}
		};

		that.strokePath = function (inParameters) {
			printCmd("strokePath", inParameters);
			if (inParameters !== undefined) {
				m_prints += " : color[";
				m_prints += inParameters.color.print();
				m_prints += "], width[";
				m_prints += inParameters.width.toString();
				if (inParameters.endCaps) {
					m_prints += "], endCaps[";
					m_prints += inParameters.endCaps;
				}
				if (inParameters.joins) {
					m_prints += "], joins[";
					m_prints += inParameters.joins;
				}
				if (inParameters.miterLimit) {
					m_prints += "], miterLimit[";
					m_prints += inParameters.miterLimit;
				}
				if (inParameters.dashes) {
					m_prints += "], dashes[";
					m_prints += inParameters.dashes.toString();
				}
				m_prints += "], path[";
				m_prints += inParameters.path.print();
				m_prints += "]";
			}
		};

		that.print = function () {
			var cmdPlayer = new CommandPlayer(m_graphics);
			m_prints = "";
			cmdPlayer.play(this);
			return m_prints;
		};

		return that;
	}
	
	utils.mixin(Graphic, {
		fillRect : function (inColor, inLTWH) {
			var newPath = new Path();

			newPath.addRect(inLTWH);
			this.fillPath(inColor, newPath);
		},

		fillPath : function (inColor, inPath) {
			this.pGraphics.commands.push({ cmdName : "fillPath", parameters : {path : inPath.clone(), color : inColor.clone() } });
		},

		//inEndCaps <object> Optional. Indicates the type of caps to use at the end of lines. One of butt, round, or square. Defaults to "butt". Also accepts the values 0 (butt), 1 (round), and 2 (square) for use with the tiny API.
		//inJoins <object> Optional. Specifies the type of joints that should be used where two lines meet. One of bevel, round, or miter. Defaults to "miter". Also accepts the values 0 (miter), 1 (round), and 2 (bevel) for use with the tiny API.
		//inMiterLimit <object> Optional. If joints is set to "miter", then you can specify a miter limit ratio which controls at what point a mitered joint will be clipped.
		//inDashes <array> Optional. on/off dash numbers
		strokeRect : function (inColor, inLTWH, inWidth, inEndCaps, inJoins, inMiterLimit, inDashes) {
			var newPath = new Path();

			newPath.addRect(inLTWH);
			this.strokePath(inColor, newPath, inWidth, inEndCaps, inJoins, inMiterLimit, inDashes);
		},

		strokePath : function (inColor, inPath, inWidth, inEndCaps, inJoins, inMiterLimit, inDashes) {
			this.pGraphics.commands.push({ cmdName : "strokePath", parameters : {path : inPath.clone(), color : inColor.clone(), width : inWidth,
																					endCaps : inEndCaps, joins : inJoins, miterLimit : inMiterLimit, dashes : inDashes} });
		},

		forEachPathNamed : function (inPathName, inFunc, inFindFirstOnly) {
			var cmds = this.getCommands(), i, p, newPath, result;

			inFindFirstOnly = inFindFirstOnly || false;

			// Replace all paths with this name.
			for (i = 0; i < cmds.length; i += 1) {
				p = cmds[i].parameters.path;
				if (p.getName() === inPathName) {
					if (inFunc) {
						newPath = inFunc(cmds[i].parameters.path);
					}
					if (newPath) {
						cmds[i].parameters.path = newPath;
					}
					if (inFindFirstOnly) {
						result = cmds[i].parameters.path;
						break;
					}
				}
			}

			return result;
		},

		forEachPath : function (lambda) {
			var i, n,
				cmds = this.getCommands(), 
				path;

			n = cmds.length;
			for (i = 0; i < n; i += 1) {
				path = cmds[i].parameters.path;
				lambda(path);
			}
		},

		findPath : function (inPathName) {
			var cmds = this.getCommands(), i, p;

			// Replace all paths with this name.
			for (i = 0; i < cmds.length; i += 1) {
				p = cmds[i].parameters.path;
				if (p.getName() === inPathName) {
					return p;
				}
			}
			return null;
		},

		setPath : function (inPathName, inPath) {
			this.forEachPathNamed(inPathName, function() { return inPath; });
			this.callChangeNotifiers("graphicChanged");
		},

		updatePaths : function (inPathUpdateFunction) {
			var cmds = this.getCommands(), i, p;

			// Replace all paths with this name.
			for (i = 0; i < cmds.length; i += 1) {
				p = inPathUpdateFunction(cmds[i].parameters.path);
				cmds[i].parameters.path = p;
			}

			this.callChangeNotifiers("graphicChanged");
		},

		clone : function (clone_children, other) {
			var result = other;

			if (result) {
				// init
				Graphic.call(result);
			} else {
				// alloc and init
				result = new Graphic();
			}

			// clone node state
			Node_clone.call(this, clone_children, result);
			// deep copy graphic state
			result.pGraphics = utils.clone(true, {}, this.pGraphics);

			return result;
		},

		getCommands : function () {
			return this.pGraphics.commands;
		},


		print : function () {
			var player = new PrintPlayer(this);
			return player.print();
		}

	});

	return Graphic;
});
